% The Doppler Effect
%
% This example demonstrates the doppler effect in which a stationary sensor
% point records a shift in frequency as a moving source travels past. It
% builds on the Monopole Point Source In A Homogeneous Propagation Medium
% Example.   
%
% author: Bradley Treeby
% date: 23rd December 2010
% last update: 9th February 2011
%  
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
% Copyright (C) 2009, 2010, 2011 Bradley Treeby and Ben Cox

% This file is part of k-Wave. k-Wave is free software: you can
% redistribute it and/or modify it under the terms of the GNU Lesser
% General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
% 
% k-Wave is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
% FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
% more details. 
% 
% You should have received a copy of the GNU Lesser General Public License
% along with k-Wave. If not, see <http://www.gnu.org/licenses/>. 

clear all;

% =========================================================================
% SIMULATION
% =========================================================================

% create the computational grid
Nx = 128;           % [pixels]
Nz = Nx/2;          % [pixels]
dx = 20e-3/Nx;    	% [m]
dz = dx;            % [m]
pml_size = 20;      % [pixels]
kgrid = makeGrid(Nx, dx, Nz, dz);

% define the properties of the propagation medium
medium.sound_speed = 1500;  % [m/s]
medium.alpha_coeff = 0.75;  % [dB/(MHz^y cm)]
medium.alpha_power = 1.5; 

% set the velocity of the moving source
source_vel = 150;           % [m/s]

% set the relative z-position between the source and sensor
source_sensor_z_distance = 5; % [pixels]

% manually create the time array
t_end = (Nx - 2*pml_size - 2)*dx / source_vel;
dt = 20e-9;
kgrid.t_array = 0:dt:t_end;

% define a single time varying sinusoidal source
source_freq = 0.75e6;       % [MHz]
source_mag = 3;
source_pressure = source_mag*sin(2*pi*source_freq*kgrid.t_array);

% smooth the source
source_pressure = filterTimeSeries(kgrid, medium, source_pressure);

% define a line of source elements
source_z_pos = 5;           % [pixels]
source.p_mask = zeros(Nz, Nx);
source.p_mask(end - pml_size - source_z_pos, 1 + pml_size:end - pml_size) = 1;

% preallocate an empty pressure source matrix
num_source_positions = sum(source.p_mask(:));
source.p = zeros(num_source_positions, length(kgrid.t_array));

% move the source along the source mask by interpolating the pressure
% series between the source elements
sensor_index = 1;
t_index = 1;
while t_index < length(kgrid.t_array) && sensor_index < num_source_positions - 1
    
    % check if the source has moved to the next pair of pixels
    if kgrid.t_array(t_index) > (sensor_index*dx/source_vel)
        sensor_index = sensor_index + 1;
    end    
    
    % calculate the position of source in between the two current pixels
    exact_pos = (source_vel*kgrid.t_array(t_index));
    discrete_pos = sensor_index*dx;
    pos_ratio = (discrete_pos - exact_pos) ./ dx;
    
    % update the pressure at the two current pixels using linear
    % interpolation
    source.p(sensor_index, t_index) = pos_ratio*source_pressure(t_index);
    source.p(sensor_index + 1, t_index) = (1 - pos_ratio)*source_pressure(t_index);
    
    % update the time index
    t_index = t_index + 1;
end

% define a single sensor element
sensor.mask = zeros(Nz, Nx);
sensor.mask(end - pml_size - source_z_pos - source_sensor_z_distance, Nx/2) = 1;

% run the simulation
[sensor_data, field_data] = kspaceFirstOrder2D(kgrid, medium, source, sensor, 'PlotPML', false);

% compute the number of time steps before any sound is recorded at the
% sensor position
source_lag = round(sqrt((Nx/2 - pml_size)^2 + source_sensor_z_distance^2)*dx/(medium.sound_speed*dt)) + 50; % [time steps]

% calculate the observed source frequency during the approach and depart
time_steps = 1500;
[f, approach_as] = spectrum(sensor_data(source_lag:source_lag + time_steps), 1/dt, 'Window', 'Hanning');
[f, retreat_as] = spectrum(sensor_data(end - time_steps:end), 1/dt, 'Window', 'Hanning');

% =========================================================================
% VISUALISATION
% =========================================================================

% plot the moving pressure field
figure;
subplot(2, 1, 1), imagesc(source.p);
colormap(getColorMap);
ylabel('Sensor Position');
xlabel('Time Step');
title('Input Pressure Signal');
subplot(2, 1, 2), plot(sum(source.p));
ylabel('Pressure [au]');
xlabel('Time Step');
title('Sum Of Input Pressure Signal Across All Sensor Positions');

% plot the simulated sensor data
figure;
[t_sc scale prefix] = scaleSI(max(kgrid.t_array(:)));
plot(kgrid.t_array*scale, sensor_data, 'r-');
xlabel(['Time [' prefix 's]']);
ylabel('Signal Amplitude');
axis tight;
title('Sensor Pressure Signal');

% plot the frequency content of the signal as the source approaches the
% sensor and as it retreats
figure;
[f_sc scale prefix] = scaleSI(max(f(:)));
plot(f*scale, approach_as, 'b-', f*scale, retreat_as, 'r-');
xlabel(['Frequency [' prefix 'Hz]']);
ylabel('Amplitude Spectrum')
axis([0 4 0 0.2])
title('Spectra During Approach and Retreat')
legend('approach','retreat')
